#ifndef LINT
static char *rcsid="$Id$";
#endif

/*
	WHAT IS IT:
		modularized contentfilter for Zmailer
	COPYRIGHT:
		(c) 2003-2005 Eugene G. Crosser <crosser@average.org>
	LICENSE:
		The same set as apply to Zmailer code
*/

#include "config.h"

#include <sys/types.h>
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef STDC_HEADERS
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

#if defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR)
# include <pthread.h>
#endif /* defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR) */

#include <zmscanner.h>
#include <clamav.h>

#include "clamav_config.h"
#include "report.h"

static int
loaddb(clamav_cfg_t *cfg)
{
	int rc;
	int virnum=0;

	DPRINT(("loaddb called\n"));
	if (cfg->root) {
		cl_free(cfg->root);
		cfg->root=NULL;
	}
	rc=cl_loaddbdir(cfg->finaldbdir,&cfg->root,&virnum);
	DPRINT(("loaddb: loaddbdir rc=%d\n",rc));
	if (rc) {
		ERRLOG((LOG_ERR,"clamav: loaddbdir(\"%s\",...) error: %s",
					cfg->finaldbdir,cl_strerror(rc)));
		return 1;
	}
	rc=cl_build(cfg->root);
	DPRINT(("loaddb: build rc=%d\n",rc));
	if (rc) {
		ERRLOG((LOG_ERR,"clamav: build error: %s",
					cl_strerror(rc)));
		return 1;
	}
	
	ERRLOG((LOG_INFO,"clamav: loaded %d virus signatures",virnum));
	return 0;
}

#if defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR)

static int count=0;
static pthread_cond_t count_cond=PTHREAD_COND_INITIALIZER;
static pthread_mutex_t count_mutex=PTHREAD_MUTEX_INITIALIZER;

static int standby=0;
static pthread_cond_t standby_cond=PTHREAD_COND_INITIALIZER;
static pthread_mutex_t standby_mutex=PTHREAD_MUTEX_INITIALIZER;

void
pre_scan(void)
{
	pthread_mutex_lock(&standby_mutex);
	while (standby) {
		DPRINT(("pre_scan: waiting on standby flag\n"));
		pthread_cond_wait(&standby_cond,&standby_mutex);
		DPRINT(("pre_scan: standby_cond signalled\n"));
	}
	pthread_mutex_lock(&count_mutex);
	count++;
	pthread_mutex_unlock(&count_mutex);
	pthread_mutex_unlock(&standby_mutex);
}

void
post_scan(void)
{
	pthread_mutex_lock(&count_mutex);
	count--;
	if (count <= 0) {
		pthread_cond_signal(&count_cond);
	}
	pthread_mutex_unlock(&count_mutex);
}

static void
lock_load(clamav_cfg_t *cfg)
{
	pthread_mutex_lock(&standby_mutex);
	standby=1;
	pthread_mutex_unlock(&standby_mutex);
	DPRINT(("refresher: standby flag set, wait for count to drop\n"));
	pthread_mutex_lock(&count_mutex);
	while (count) {
		DPRINT(("refresher: waiting for count to become zero\n"));
		pthread_cond_wait(&count_cond,&count_mutex);
		DPRINT(("refresher: count_cond signalled\n"));
	}
	pthread_mutex_unlock(&count_mutex);

	DPRINT(("refresher: count dropped to zero, may refresh database\n"));
	/* do real job here */
	(void)loaddb(cfg);
	/* real job done */
	DPRINT(("refresher: refresh complete, reset standby flag\n"));

	pthread_mutex_lock(&standby_mutex);
	standby=0;
	pthread_cond_signal(&standby_cond);
	pthread_mutex_unlock(&standby_mutex);
}

static void *
refresher(void *priv)
{
	clamav_cfg_t *cfg=(clamav_cfg_t *)priv;
	struct cl_stat dbstat;

	memset(&dbstat,0,sizeof(dbstat));
	cl_statinidir(cfg->finaldbdir,&dbstat);
	while (1) {
		sleep(10);
		if (cl_statchkdir(&dbstat)) {
			lock_load(cfg);
			cl_statfree(&dbstat);
			cl_statinidir(cfg->finaldbdir,&dbstat);
		}
	}
}

int
clamav_refreshdb(clamav_cfg_t *cfg)
{
	int rc;
	static pthread_t refresh_thread;
	static pthread_attr_t attr;

	DPRINT(("clamav_refreshdb thread starting\n"));
	rc=loaddb(cfg);
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
	pthread_create(&refresh_thread,&attr,refresher,(void *)cfg);
	return rc;
}

#else /* defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR) */

void
pre_scan(void)
{}

void
post_scan(void)
{}

int
clamav_refreshdb(clamav_cfg_t *cfg)
{
	DPRINT(("clamav_refreshdb single-run\n"));
	return loaddb(cfg);
}

#endif /* defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR) */
